﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using DataAccess;
using System.Data;
using ext = Ext.Net;

public partial class ucPatPortalAccount : System.Web.UI.UserControl
{
    public BaseMaster BaseMstr { get; set; }

    protected void Page_Load(object sender, EventArgs e)
    {
        
    }

    /// <summary>
    /// US:6967 initialize web user control; 
    /// loads patient's account details if an account exists
    /// </summary>
    public void InitializeControl() 
    {
        //check if patient account exists
        bool bHasPortalAccount = PatientAccountExists();
        if (bHasPortalAccount)
        {
            // load patient account details
            LoadPatientAccount();

            //hide password fields
            //they will only be shown when resetting the password
            txtPassword.Clear();
            txtPassword.Hide();

            txtVerifyPassword.Clear();
            txtVerifyPassword.Hide();
            
            //show password reset button
            btnResetPassword.Show();
        }
        else {
            //hide password reset button
            btnResetPassword.Hide();

            //show password fields
            //screen is ready to create a new account
            txtPassword.Show();
            txtVerifyPassword.Show();

            //disable account status checkboxes
            chkLocked.Disabled = true;
            chkInactive.Disabled = true;
        }

        // toggle read only status for username field
        // depending on the existance of a portal account
        txtUsername.ReadOnly = bHasPortalAccount;

        //toggle lable in update/create account button
        btnUpdateAccount.Text = bHasPortalAccount ? "Update Account" : "Create Account";

        //sets reset password mode to off
        htxtIsChangePwd.Value = "0";

        //change reset password button label
        btnResetPassword.Text = "Reset Password";
        btnResetPassword.Icon = ext.Icon.Pencil;

        return;
    }

    /// <summary>
    /// US:6967 prepares the account's form to perform a password reset
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void ResetPassword(object sender, ext.DirectEventArgs e)
    {
        if (PatientAccountExists())
        {
            if (htxtIsChangePwd.Value.Equals("0"))
            {
                //clears and shows password field
                txtPassword.Clear();
                txtPassword.Show();

                //clears and shows password's verify field
                txtVerifyPassword.Clear();
                txtVerifyPassword.Show();

                //sets mode to reset password
                htxtIsChangePwd.Value = "1";

                //change button label
                btnResetPassword.Text = "Cancel Reset Password";
                btnResetPassword.Icon = ext.Icon.Cancel;
            }
            else
            {
                //clears and hides password field
                txtPassword.Clear();
                txtPassword.Hide();

                //clears and hides password's verify field
                txtVerifyPassword.Clear();
                txtVerifyPassword.Hide();

                //sets mode to reset password
                htxtIsChangePwd.Value = "0";

                //change button label
                btnResetPassword.Text = "Reset Password";
                btnResetPassword.Icon = ext.Icon.Pencil;
            }
        }
    }

    /// <summary>
    /// US:6967 submits account's form for updating in the central data base
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void UpdatePortalAccount(object sender, ext.DirectEventArgs e)
    {
        bool bHasPortalAccount = PatientAccountExists();
        bool bIsResetPwd = Convert.ToBoolean(Convert.ToInt16(htxtIsChangePwd.Value));
        
        //first, validate account form
        string strValidationMsg = String.Empty;
        if (ValidateAccountForm(out strValidationMsg))
        {
            
            // ****************************************************************************
            // ONLY UPDATE ACCOUNT STATUS
            // ****************************************************************************

            if (bHasPortalAccount && !bIsResetPwd)
            {
                if (UpdateAccountStatus())
                {
                    InitializeControl();
                    BaseMstr.ShowSystemFeedback("Patient Account status was updated successfully.", "Account updated");
                    return;
                }
                else
                {
                    BaseMstr.ShowSystemFeedback(BaseMstr.StatusComment, "Error Updating Account");
                    return;
                }
            }

            // ****************************************************************************
            // CREATE NEW ACCOUNT
            // ****************************************************************************

            if (!bHasPortalAccount)
            {
                if (CreatePatientAccount())
                {
                    InitializeControl();
                    BaseMstr.ShowSystemFeedback("Patient Account was created successfully.", "Account created");
                    return;
                }
                else
                {
                    BaseMstr.ShowSystemFeedback(BaseMstr.StatusComment, "Error Creating Account");
                    return;
                }
            }
           

            // ****************************************************************************
            // UPDATE ACCOUNT (PASSWORD & STATUS)
            // ****************************************************************************

            if (bHasPortalAccount && bIsResetPwd)
            {
                if (UpdateAccountPassword())
                {
                    InitializeControl();
                    BaseMstr.ShowSystemFeedback("Account password was updated successfully.", "Password updated");
                    return;
                }
                else
                {
                    BaseMstr.ShowSystemFeedback(BaseMstr.StatusComment, "Error updating password");
                    return;
                }
            }
        }
        else 
        {
            BaseMstr.ShowSystemFeedback(strValidationMsg, "Account Validation Errors");
        }
    }

    /// <summary>
    /// US:6967 validates the account's form before submiting for update
    /// </summary>
    /// <returns></returns>
    protected bool ValidateAccountForm(out string strValidationComment) {
        bool bValidate = true;

        bool bHasPortalAccount = PatientAccountExists();
        bool bIsResetPwd = Convert.ToBoolean(Convert.ToInt16(htxtIsChangePwd.Value));

        strValidationComment = String.Empty;
        List<string> lstErrors = new List<string>();

        //bypass validation if only changing account status
        if (bHasPortalAccount && !bIsResetPwd)
        {
            return true;
        }

        CSec secx = new CSec();

        // VERIFY USER NAME ------------------------------------------
        if (!String.IsNullOrEmpty(txtUsername.Text))
        {
            if (!bHasPortalAccount)
            {
                //check if username is not already taken
                if (secx.UserNameExists(BaseMstr, txtUsername.Text))
                {
                    bValidate = false;
                    lstErrors.Add("Please choose a different username.");
                } 
            }

        }
        else {
            bValidate = false;
            lstErrors.Add("Please enter a username.");
        }

        // VERIFY PASSWORD ------------------------------------------
        if (String.IsNullOrEmpty(txtPassword.Text))
        {
            bValidate = false;
            lstErrors.Add("Please enter a password.");
        }
        else {
            //check that password fields are the same
            if (!txtPassword.Text.Equals(txtVerifyPassword.Text))
            {
                bValidate = false;
                lstErrors.Add("Password and verify password must match.");
            }
            else {
                //check all the account rules
                if (!secx.ValidateUserAccountRules(BaseMstr,
                                                   txtUsername.Text,
                                                   txtPassword.Text))
                {
                    bValidate = false;
                    lstErrors.Add(BaseMstr.StatusComment);
                }
            }
        }
        
        // -----------------------------------------------------------------
        strValidationComment += "~#(ul)#~";
        strValidationComment += TransformErrorMessage(lstErrors);
        strValidationComment += "~#(/ul)#~";
        return bValidate;
    }

    /// <summary>
    /// US:6967 checks if the patient portal account exists
    /// </summary>
    /// <returns></returns>
    protected bool PatientAccountExists()
    {
        CSec sec = new CSec();
        DataSet secDataChk = sec.CheckPatientFXUserRecDS(BaseMstr);

        //load all of the patient's available fields
        if (secDataChk != null)
        {
            foreach (DataTable secTable in secDataChk.Tables)
            {
                foreach (DataRow secRow in secTable.Rows)
                {
                    if (!secRow.IsNull("FXUSERCOUNT"))
                    {
                        long lFXUserCount = Convert.ToInt32(secRow["FXUSERCOUNT"]);
                        if (lFXUserCount > 0)
                        {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    /// <summary>
    /// US:6967 loads the patient's account details
    /// </summary>
    protected void LoadPatientAccount() {
        CDataUtils utils = new CDataUtils();
        CSec sec = new CSec();

        DataSet dsSecData = new DataSet();

        //attempt to grab the user's profile
        dsSecData = sec.GetPatientFXUsernamePasswordDS(BaseMstr);

        //load FXUser Username and Password fields
        if (dsSecData != null)
        {
            //disable account status checkboxes
            chkLocked.Disabled = false;
            chkInactive.Disabled = false;
            
            txtUsername.Text = sec.dec(utils.GetStringValueFromDS(dsSecData, "USER_NAME"), "");
            chkLocked.Checked  = Convert.ToBoolean(utils.GetLongValueFromDS(dsSecData, "IS_LOCKED"));
            chkInactive.Checked = Convert.ToBoolean(utils.GetLongValueFromDS(dsSecData, "IS_INACTIVE"));
        }
    }

    /// <summary>
    /// US:6967 creates a patient account record in the central database
    /// </summary>
    /// <returns></returns>
    protected bool CreatePatientAccount() {
        CSec sec = new CSec();
        long lFXUserID = 0;

        return sec.InsertPatientFXUser(BaseMstr,
                                             BaseMstr.SelectedPatientID,
                                             txtUsername.Text,
                                             txtPassword.Text,
                                             chkLocked.Checked,
                                             chkInactive.Checked,
                                             out lFXUserID);
    }

    /// <summary>
    /// US:6967 updates the patient portal account in the database
    /// </summary>
    /// <returns></returns>
    protected bool UpdateAccountStatus()
    {
        CSec sec = new CSec();
        long lFXUserID = getPatientFXUserID();
        return sec.UpdatePatientFXUserOptions(BaseMstr,
                                            lFXUserID,
                                            chkLocked.Checked,
                                            chkInactive.Checked);
    }

    /// <summary>
    /// US:6967 updates portal account password
    /// </summary>
    /// <returns></returns>
    protected bool UpdateAccountPassword() {
        CSec sec = new CSec();
        long lFXUserID = getPatientFXUserID();
        return sec.UpdatePatientFXUserPWD(BaseMstr,
                                        lFXUserID,
                                        txtUsername.Text,
                                        txtPassword.Text,
                                        chkLocked.Checked,
                                        chkInactive.Checked);
    }

    /// <summary>
    /// Transforms the error list (List<string>) into an HTML UL string
    /// to be displayed in the System Feedbacks window
    /// </summary>
    /// <param name="lstErrors"></param>
    /// <returns></returns>
    protected string TransformErrorMessage(List<string> lstErrors)
    {
        string[] strErrors = lstErrors.ToArray();
        string strOutput = String.Empty;

        foreach (string strE in strErrors)
        {
            strOutput += String.Format("~#(li)#~{0}~#(/li)#~", strE);
        }

        return strOutput;
    }

    /// <summary>
    /// US:6967 gets the user id for the selected patient
    /// </summary>
    /// <returns></returns>
    protected long getPatientFXUserID()
    {
        long Value = 0;

        CSec fxSec = new CSec();

        DataSet SecSet = new DataSet();
        SecSet = fxSec.GetPatientFXUserIdDS(BaseMstr);

        //load all of the user's available fields
        if (SecSet != null)
        {
            foreach (DataTable secTable in SecSet.Tables)
            {
                foreach (DataRow secRow in secTable.Rows)
                {
                    if (!secRow.IsNull("FX_USER_ID"))
                    {
                        Value = Convert.ToInt64(secRow["FX_USER_ID"]);
                    }
                }
            }
        }

        return Value;
    }
}